//------------------------------------------//
// filter.cpp                               //
//------------------------------------------//

//must include this one
#include "filter.h"

//headers specific to this filter
#include <Slider.h>
#include <Button.h>
#include "stdio.h"		//for strcpy
#include "stdlib.h"		//for atoi
#include "malloc.h"

//Constants used in filter configuration window
const char FILTER_NAME[] 		= "Brightness";
const char WINDOW_TITLE[] 		= "Brightness Configure";
const char AUTHOR_NAME[] 		= "Sean Long";
const char VERSION_NAME[] 		= "1.00";
const char DESCRIPTION_NAME[]	= "Brighten or darken image.";
BRect	   CONFIG_WINDOW_SIZE(0,0,170,100);
BPoint	   WINDOW_TOP_LEFT(50,50);

bool	   windowopen=false;
BPath	   FilterSavePath;

//BMessages for Configuration window (please start at B_SPECIFIERS_END+50)
#define VALUE_CHANGED 				B_SPECIFIERS_END+50
#define SET_CENTER					B_SPECIFIERS_END+51


//filter parameters
float 	brighten_fac=100;


class FCView;
class FCWindow;

bool 	LoadPrefs();
void 	SavePrefs(FCView *handle);

//Our window object
FCWindow		*configWin;

//view class used in the configuration window
class FCView : public BView
{
	public:
		FCView(BRect frame,char *name);
		~FCView();
		
		BSlider			*BrightenSlide;	
		BButton			*SetCenter;
		
	private:
	
};

//class used for configuration window
class FCWindow : public BWindow
{
	public:
		FCWindow(BRect frame,char *title);
		~FCWindow();
		
		bool		QuitRequested();
	private:
		FCView		*configView;
		void		MessageReceived(BMessage *message);		
	
};

//------------------------------------------
// Brightness

//construcor
Filter::Filter( image_id id )
{
	mId			= id;
 	Author 		= NULL;
 	Version		= NULL;
 	Description = NULL;
 	configWin	= NULL;

	//set path to save/load filter prefs at
	find_directory(B_USER_SETTINGS_DIRECTORY,&FilterSavePath);
	FilterSavePath.Append("ImageGrinder/filters");
	FilterSavePath.Append(FILTER_NAME);	

	Author = new char[strlen(AUTHOR_NAME)+1];
	strcpy(Author,AUTHOR_NAME);
	
	Version = new char[strlen(VERSION_NAME)+1];
	strcpy(Version,VERSION_NAME);

	Description = new char[strlen(DESCRIPTION_NAME)+1];
	strcpy(Description,DESCRIPTION_NAME);
		
}

Filter::~Filter()
{
	if(Author)		delete Author;
	if(Version)		delete Version;
	if(Description)	delete Description;
	if(configWin)	delete configWin;

}

void Filter::Configure()
{	
	if(windowopen==false)
	{
		configWin = new FCWindow(CONFIG_WINDOW_SIZE,WINDOW_TITLE);					
		windowopen=true;
	}
}	

void Filter::Process(const BBitmap *src, BBitmap **dest)
{
	//the actual function that manipulates the bitmap		  
	LoadPrefs();
	
	if(src == NULL) return;
	
	BRect bounds = src->Bounds();
	color_space cs = src->ColorSpace();
	if (cs != B_RGB32 && cs != B_RGBA32) 
	{
		return;
	}
	
	if(brighten_fac==100)
	{
		//special case so format conversions goes faster
		//copy bits from src bitmap into dest bitmap & return
		(*dest) = new BBitmap(bounds,cs);
		if(*dest != NULL)
			memcpy((*dest)->Bits(), src->Bits(), src->BitsLength());			
		return;
	}	
	
	*dest = new BBitmap(bounds,cs);	
	if (*dest == NULL) 
	{	
		return;
	}

	register long drows, dcols, srows, scols;
	register unsigned long *spixptr, *spix1;
	register unsigned char *dpixptr, *dpix;
	
	register unsigned long slb, dlb;
	register long i, j;
	register unsigned char *a;

	srows = src->Bounds().IntegerHeight()+1;
	scols = src->Bounds().IntegerWidth()+1;
	drows = (*dest)->Bounds().IntegerHeight();
	dcols = (*dest)->Bounds().IntegerWidth();
		
	// Get values from image
	dpixptr = (unsigned char *)(*dest)->Bits();
	dpix = (unsigned char *)dpixptr;
	spixptr = (unsigned long *)src->Bits();
	slb = src->BytesPerRow()/ 4;
	dlb = (*dest)->BytesPerRow();

	float ar,ag,ab;

	float mult_fac = brighten_fac/100;

	//mirror horizontally 	
	for(i=0; i <= drows;i++)
	{
		spix1 = spixptr + i*slb;
		for (j =0; j <= dcols; j++)
		{
			a = (unsigned char *)(spix1 + j);
			
			ar = (float)a[0]*(mult_fac);
			ag = (float)a[1]*(mult_fac);
			ab = (float)a[2]*(mult_fac);

			dpix[0] = (unsigned char) (ar >255) ? 255 : ar;
			dpix[1] = (unsigned char) (ag >255) ? 255 : ag;
			dpix[2] = (unsigned char) (ab >255) ? 255 : ab;
			dpix += 4;
		}		
		// Advance to the next destination line
		dpix = (dpixptr += dlb);
	}	

	
}

//------------------------------------------
// Configuration Window

FCWindow::FCWindow(BRect frame,char *title)
	:BWindow(frame,title,B_TITLED_WINDOW,B_NOT_ZOOMABLE | B_NOT_RESIZABLE,B_CURRENT_WORKSPACE)
{
	configView = new FCView(Bounds(),title);
	AddChild(configView);
	MoveTo(WINDOW_TOP_LEFT);
	Show();
}

FCWindow::~FCWindow()
{
	
}

bool FCWindow::QuitRequested()
{
	//get the window position
	BRect	top_left;
	top_left = Frame();
	WINDOW_TOP_LEFT.x = top_left.left;
	WINDOW_TOP_LEFT.y = top_left.top;		
	
	SavePrefs(configView);
	windowopen=false;
	Quit();
	return true;	
}

void FCWindow::MessageReceived(BMessage *message)
{

	switch(message->what)
	{	
		case VALUE_CHANGED:
			SavePrefs(configView);
			break;	
		
		case SET_CENTER:
			brighten_fac  = 100;
			configView->BrightenSlide->SetPosition(100.0/255.0);  
			SavePrefs(configView);
			break;	
   		default:
			BWindow::MessageReceived(message);
			break;
     
	}
	
}

FCView::FCView(BRect frame,char *name)
	:BView(frame, name, B_FOLLOW_ALL, B_WILL_DRAW | B_PULSE_NEEDED)
{

	//load the saved prefs if any
	LoadPrefs();

	//initialize all pointers to NULL
	BrightenSlide		= NULL;
	SetCenter			= NULL;
	
	SetViewColor(216,216,216);
	
	//draw configuration panel here

	//Layout the view
	BrightenSlide = new BSlider(BRect(10,10,160,45), "Brightness", "Brightness",new BMessage(VALUE_CHANGED), 0, 255);
	rgb_color 	blue ={5,5,255,55};
	BrightenSlide->UseFillColor(true,&blue);
	BrightenSlide->SetStyle(B_TRIANGLE_THUMB);
	BrightenSlide->SetLimitLabels("Darken", "Brighten");
	BrightenSlide->SetValue(brighten_fac);
	BrightenSlide->SetSnoozeAmount(100000);	//so messages will be sent a little slower
	AddChild(BrightenSlide);	
	
	SetCenter = new BButton(BRect(10,70,160,90), "SetCenter", "No change", new BMessage(SET_CENTER), B_FOLLOW_NONE);
	SetCenter->SetFontSize(10.0);
	SetCenter->ResizeToPreferred();
	AddChild(SetCenter);		
		
}

FCView::~FCView()
{
	
	//remove all of this views children
	BView	*aView = NULL;
	bool	result = false;

	if((aView = ChildAt(0)))
	{
		while(aView != NULL) 
		{
			result = RemoveChild(aView);
			if(result) 
			{
				delete aView;
				aView = NULL;
			}
			aView = ChildAt(0);
		}
		aView = NULL;
	}
		
}

bool LoadPrefs()
{
	//load filter prefs here prefered method like below	
	
	//try opening the saved pref for this filter
	BFile	file(FilterSavePath.Path(),B_READ_ONLY);
	
	if(file.InitCheck() != B_OK)
	{
		//no prefs file found defaults will be used 
		return false;
	}
	else
	{
		//get the info from file
		char buffer[10];
		
		ssize_t result=0;
		
		result = file.ReadAttr("WindowTopLeftPointX",B_STRING_TYPE,0,buffer,sizeof(buffer));
		if(result > 0) WINDOW_TOP_LEFT.x = atof(buffer);
		
		result = file.ReadAttr("WindowTopLeftPointY",B_STRING_TYPE,0,buffer,sizeof(buffer));
		if(result > 0) WINDOW_TOP_LEFT.y = atof(buffer);			
		
		result = file.ReadAttr("BrightnessFactor",B_STRING_TYPE,0,buffer,10);
		if(result > 0) brighten_fac = atof(buffer);
					
	
		return true;	
	}
	
}

void SavePrefs(FCView *handle)
{
	//save filter prefs here prefered method like below

	BPath	path;
	find_directory(B_USER_SETTINGS_DIRECTORY,&path);
	path.Append("ImageGrinder/filters");	
	BDirectory	dir(path.Path());
	BFile	file(FILTER_NAME,B_WRITE_ONLY);
	
	//create file if does not exist if it does the old gets clobbered
	dir.CreateFile(FilterSavePath.Path(),&file,false);
	
	//now save into it
	char buffer[10];
	
	sprintf(buffer,"%f",WINDOW_TOP_LEFT.x);
	file.WriteAttr("WindowTopLeftPointX",B_STRING_TYPE,0,buffer,strlen(buffer)+1);

	sprintf(buffer,"%f",WINDOW_TOP_LEFT.y);
	file.WriteAttr("WindowTopLeftPointY",B_STRING_TYPE,0,buffer,strlen(buffer)+1);		
	
	sprintf(buffer,"%f",(float)handle->BrightenSlide->Value());
	file.WriteAttr("BrightnessFactor",B_STRING_TYPE,0,buffer,strlen(buffer)+1);
	
}
